<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ResponseTemplate;
use App\Models\Group;

class ResponseTemplateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $groups = Group::where('status', 1)->get();

        $query = ResponseTemplate::where('status', 1)->with('groups');

        if ($request->has('group')) {
            $query->whereHas('groups', function ($q) use ($request) {
                $q->where('groups.id', $request->group);
            });
        }

        $responseTemplates = $query->paginate(10);

        return view('admin.response_templates.index', compact('responseTemplates', 'groups'));
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $groups = Group::where('status', 1)->get();
        $selectedGroup = $request->group;
        return view('admin.response_templates.create', compact('groups', 'selectedGroup'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'write_to' => 'required|string',
            'reason' => 'required|string',
            'stage_of_booking' => 'required|string',
            'type' => 'required|string',
            'subject' => 'required|string',
            'content' => 'nullable|string',
            'notes' => 'nullable|string',
            'groups' => 'required|array',
        ]);
        $template = ResponseTemplate::create($request->only(['title', 'write_to', 'reason', 'stage_of_booking', 'type', 'subject', 'content', 'notes']));
        $template->groups()->sync($request->groups); // Sync pivot table

        return redirect()->route('admin.response_templates.index')->with('success', 'Template created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(ResponseTemplate $responseTemplate)
    {
        $groups = Group::where('status', 1)->get();
        return view('admin.response_templates.show', compact('responseTemplate', 'groups'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ResponseTemplate $responseTemplate)
    {
        $groups = Group::where('status', 1)->get();
        return view('admin.response_templates.edit', compact('responseTemplate', 'groups'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ResponseTemplate $responseTemplate)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'write_to' => 'required|string',
            'reason' => 'required|string',
            'stage_of_booking' => 'required|string',
            'type' => 'required|string',
            'subject' => 'required|string',
            'content' => 'nullable|string',
            'notes' => 'nullable|string',
            'groups' => 'required|array',
        ]);

        $responseTemplate->update($request->only(['title', 'write_to', 'reason', 'stage_of_booking', 'type', 'subject', 'content', 'notes']));
        $responseTemplate->groups()->sync($request->groups);

        return redirect()->route('admin.response_templates.index')->with('success', 'Template updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ResponseTemplate $responseTemplate, Request $request)
    {
        $responseTemplate->delete();

        if ($request->has('group')) {
            return redirect()->route('admin.response_templates.index',['group' => $request->group])->with('success', 'Template deleted updated successfully.');
        }
        return redirect()->route('admin.response_templates.index')->with('success', 'Template deleted successfully.');
    }

    public function toggleStatus(ResponseTemplate $responseTemplate, Request $request)
    {
        $responseTemplate->status = !$responseTemplate->status;
        $responseTemplate->save();

        if ($request->has('group')) {
            return redirect()->route(['group' => $request->group])->with('success', 'Template status updated successfully.');
        }
        return redirect()->route('admin.response_templates.index')->with('success', 'Template status updated successfully.');
    }
    public function enabledResponseTemplates(Request $request)
    {
        // $groupId = $request->query('group');
        $groups =  Group::where('status', 1)->get();
        $query = ResponseTemplate::where('status', 1)->with('groups');

        if ($request->has('group')) {
            $query->whereHas('groups', function ($q) use ($request) {
                $q->where('groups.id', $request->group);
            });
        }

        $responseTemplates = $query->paginate(10);

        return view('admin.response_templates.index', compact('responseTemplates', 'groups'));
    }

    public function disabledResponseTemplates(Request $request)
    {
        // $groupId = $request->query('group');
        $groups =  Group::where('status', 1)->get();
        $query = ResponseTemplate::where('status', 0)->with('groups');

        if ($request->has('group')) {
            $query->whereHas('groups', function ($q) use ($request) {
                $q->where('groups.id', $request->group);
            });
        }

        $responseTemplates = $query->paginate(10);;

        return view('admin.response_templates.index', compact('responseTemplates', 'groups'));
    }

    public function trashed(Request $request)
    {
        // $groupId = $request->query('group');
        // $group = Group::findOrFail($groupId);
        $query = ResponseTemplate::onlyTrashed()->with('groups');

        if ($request->has('group')) {
            $query->whereHas('groups', function ($q) use ($request) {
                $q->where('groups.id', $request->group);
            });
        }

        $trashedTemplates = $query->paginate(10);;

        return view('admin.response_templates.trashed', compact('trashedTemplates'));
    }

    public function restore($id, Request $request)
    {
        $template = ResponseTemplate::onlyTrashed()->findOrFail($id);
        $template->restore();

        if ($request->has('group')) {
            return redirect()->route('admin.response_templates.index',['group' => $request->group])->with('success', 'Template restored successfully.');
        }
        return redirect()->route('admin.response_templates.index')->with('success', 'Template restored successfully.');
    }
}
